/*
 * Decompiled with CFR 0.152.
 */
package velox.api.layer1.simplified;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Path;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.commons.collections4.BidiMap;
import org.apache.commons.collections4.bidimap.DualHashBidiMap;
import quickfix.RuntimeError;
import velox.api.layer1.common.DirectoryResolver;
import velox.api.layer1.common.Log;
import velox.api.layer1.layers.strategies.interfaces.OnlineCalculatable;

class IconStorage
implements AutoCloseable {
    private static final Path TEMP_DIRECTORY = DirectoryResolver.getBookmapDirectoryByName((String)"API").resolve("SimplifiedL1ImageCache");
    private static final long MAX_INMEMORY_BUFFER = 0x400000L;
    static final long MIN_GC_SIZE = 524288L;
    static final double MIN_GC_FACTOR = 3.0;
    private Map<Integer, InMemoryIconInfo> inMemoryIconInfos = new HashMap<Integer, InMemoryIconInfo>();
    private BidiMap<Long, Hash256> hashByOffset = new DualHashBidiMap();
    private HashMap<Hash256, Integer> hashUsesCount = new HashMap();
    private final File imagesTempFolder;
    private final File storageFile;
    private final RandomAccessFile storageRandomAccessFile;
    private FileLock storageFileLock;
    private long storageRandomAccessFileNextOffset = 0L;
    private long removedBytes = 0L;
    private boolean closed = false;
    private final Cache<Long, BufferedImage> memoryCache;

    public IconStorage() {
        this(TEMP_DIRECTORY, true, TimeUnit.MINUTES.toNanos(10L));
    }

    IconStorage(Path tempDirectory, boolean cleanDirectory, long expireAfterAccessNanos) {
        this.memoryCache = CacheBuilder.newBuilder().maximumWeight(0x400000L).weigher((k, v) -> {
            BufferedImage bufferedImage = (BufferedImage)v;
            return bufferedImage.getHeight() * bufferedImage.getWidth() * 4;
        }).expireAfterAccess(expireAfterAccessNanos, TimeUnit.NANOSECONDS).build();
        try {
            this.imagesTempFolder = tempDirectory.toFile().getAbsoluteFile();
            if (!this.imagesTempFolder.exists()) {
                this.imagesTempFolder.mkdir();
            }
            if (cleanDirectory) {
                this.cleanOldFiles();
            }
            this.storageFile = File.createTempFile("icons-", null, this.imagesTempFolder).getAbsoluteFile();
            this.storageRandomAccessFile = new RandomAccessFile(this.storageFile, "rw");
            this.storageFileLock = this.storageRandomAccessFile.getChannel().tryLock();
        }
        catch (IOException e) {
            throw new RuntimeError((Throwable)e);
        }
    }

    private void cleanOldFiles() {
        for (File file : this.imagesTempFolder.listFiles()) {
            File absoluteFile = file.getAbsoluteFile();
            if (!absoluteFile.isFile()) continue;
            FileLock lock = null;
            RandomAccessFile raFile = null;
            try {
                raFile = new RandomAccessFile(absoluteFile, "rw");
                lock = raFile.getChannel().tryLock();
            }
            catch (OverlappingFileLockException overlappingFileLockException) {
            }
            catch (IOException e) {
                Log.info((String)("Failed to check lock on file " + String.valueOf(absoluteFile)), (Exception)e);
            }
            try {
                if (lock != null && lock.isValid()) {
                    lock.release();
                }
                if (raFile != null) {
                    raFile.close();
                }
            }
            catch (IOException e) {
                Log.info((String)("Failed to release lock on file " + String.valueOf(absoluteFile)), (Exception)e);
            }
            if (lock == null) continue;
            absoluteFile.delete();
        }
    }

    @Override
    public synchronized void close() {
        if (!this.closed) {
            try {
                this.storageFileLock.release();
                this.storageRandomAccessFile.close();
                boolean deleted = this.storageFile.delete();
                Log.info((String)("storageFile deletion return: " + deleted));
            }
            catch (IOException e) {
                throw new RuntimeException();
            }
            this.closed = true;
        }
    }

    public synchronized boolean isClosed() {
        return this.closed;
    }

    protected void finalize() throws Throwable {
        try {
            this.close();
        }
        catch (Throwable t) {
            Log.error((String)"Failed to finalize", (Throwable)t);
        }
        super.finalize();
    }

    public synchronized void save(IconInfo iconInfo) {
        if (this.closed) {
            throw new IllegalStateException("Closed");
        }
        if (this.inMemoryIconInfos.containsKey(iconInfo.imageId)) {
            throw new IllegalArgumentException("Icon with the same ID already exists: " + iconInfo.imageId);
        }
        BufferedImage icon = iconInfo.marker.icon;
        InMemoryIconInfo inMemoryIconInfo = new InMemoryIconInfo();
        inMemoryIconInfo.markerY = iconInfo.marker.markerY;
        inMemoryIconInfo.iconOffsetX = iconInfo.marker.iconOffsetX;
        inMemoryIconInfo.iconOffsetY = iconInfo.marker.iconOffsetY;
        inMemoryIconInfo.imageWidth = iconInfo.marker.icon.getWidth();
        inMemoryIconInfo.imageHeight = iconInfo.marker.icon.getHeight();
        byte[] imageBytes = IconStorage.getImageBytes(icon);
        Hash256 hash = IconStorage.getImageHash(imageBytes);
        Long offset = (Long)this.hashByOffset.getKey((Object)hash);
        if (offset == null) {
            try {
                offset = this.storageRandomAccessFileNextOffset;
                this.storageRandomAccessFileNextOffset += (long)imageBytes.length;
                this.storageRandomAccessFile.seek(offset);
                this.storageRandomAccessFile.write(imageBytes);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to save icon to cache", e);
            }
            this.hashByOffset.put((Object)offset, (Object)hash);
            this.hashUsesCount.put(hash, 1);
        } else {
            this.hashUsesCount.compute(hash, (h, count) -> count + 1);
        }
        inMemoryIconInfo.imageOffsetInFile = offset;
        this.memoryCache.put((Object)offset, (Object)icon);
        this.inMemoryIconInfos.put(iconInfo.imageId, inMemoryIconInfo);
    }

    private static byte[] getImageBytes(BufferedImage image) {
        int[] imageInts = image.getRGB(0, 0, image.getWidth(), image.getHeight(), null, 0, image.getWidth());
        ByteBuffer byteBuffer = ByteBuffer.allocate(imageInts.length * 4);
        IntBuffer intBuffer = byteBuffer.asIntBuffer();
        intBuffer.put(imageInts);
        byte[] imageBytes = byteBuffer.array();
        return imageBytes;
    }

    private static Hash256 getImageHash(byte[] imageBytes) {
        return new Hash256(imageBytes);
    }

    public synchronized IconInfo load(int iconId) {
        BufferedImage icon;
        if (this.closed) {
            throw new IllegalStateException("Closed");
        }
        InMemoryIconInfo inMemoryIconInfo = this.inMemoryIconInfos.get(iconId);
        try {
            icon = (BufferedImage)this.memoryCache.get((Object)inMemoryIconInfo.imageOffsetInFile, () -> {
                BufferedImage image = this.loadFromDisk(inMemoryIconInfo);
                return image;
            });
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
        IconInfo iconInfo = new IconInfo();
        iconInfo.imageId = iconId;
        iconInfo.marker = new OnlineCalculatable.Marker(inMemoryIconInfo.markerY, inMemoryIconInfo.iconOffsetX, inMemoryIconInfo.iconOffsetY, icon);
        return iconInfo;
    }

    private BufferedImage loadFromDisk(InMemoryIconInfo inMemoryIconInfo) throws IOException {
        this.storageRandomAccessFile.seek(inMemoryIconInfo.imageOffsetInFile);
        ByteBuffer byteBuffer = ByteBuffer.allocate(this.getImageSizeInBytes(inMemoryIconInfo));
        this.storageRandomAccessFile.readFully(byteBuffer.array());
        IntBuffer asIntBuffer = byteBuffer.asIntBuffer();
        int[] imageInts = new int[inMemoryIconInfo.imageWidth * inMemoryIconInfo.imageHeight];
        asIntBuffer.get(imageInts);
        BufferedImage image = new BufferedImage(inMemoryIconInfo.imageWidth, inMemoryIconInfo.imageHeight, 2);
        image.setRGB(0, 0, image.getWidth(), image.getHeight(), imageInts, 0, image.getWidth());
        return image;
    }

    private int getImageSizeInBytes(InMemoryIconInfo inMemoryIconInfo) {
        return inMemoryIconInfo.imageWidth * inMemoryIconInfo.imageHeight * 4;
    }

    public synchronized void remove(int iconId) {
        if (this.closed) {
            throw new IllegalStateException("Closed");
        }
        InMemoryIconInfo removedInfo = this.inMemoryIconInfos.remove(iconId);
        Hash256 hash = (Hash256)this.hashByOffset.get((Object)removedInfo.imageOffsetInFile);
        Integer updatedHashUses = this.hashUsesCount.compute(hash, (h, count) -> count == 1 ? null : Integer.valueOf(count - 1));
        if (updatedHashUses == null) {
            this.hashByOffset.remove((Object)removedInfo.imageOffsetInFile);
            this.memoryCache.invalidate((Object)removedInfo.imageOffsetInFile);
            this.removedBytes += (long)this.getImageSizeInBytes(removedInfo);
            this.collectGarbage();
        }
    }

    private void collectGarbage() {
        if (this.inMemoryIconInfos.isEmpty()) {
            if (!this.hashByOffset.isEmpty() || !this.hashUsesCount.isEmpty()) {
                throw new RuntimeException("No icons in storage but size of helper structures is non-zero");
            }
            this.storageRandomAccessFileNextOffset = 0L;
            this.removedBytes = 0L;
            this.memoryCache.invalidateAll();
        }
        if (this.removedBytes >= 524288L && (double)this.removedBytes / (double)this.storageRandomAccessFileNextOffset >= 0.6666666666666666) {
            this.storageRandomAccessFileNextOffset = 0L;
            this.removedBytes = 0L;
            this.memoryCache.invalidateAll();
            HashMap offsetMap = new HashMap();
            this.inMemoryIconInfos.values().stream().sorted((a, b) -> Long.compare(a.imageOffsetInFile, b.imageOffsetInFile)).forEach(inMemoryIconInfo -> {
                Long remappedOffset = (Long)offsetMap.get(inMemoryIconInfo.imageOffsetInFile);
                if (remappedOffset == null) {
                    int bytesToMove = this.getImageSizeInBytes((InMemoryIconInfo)inMemoryIconInfo);
                    try {
                        long imageOffsetInFile = inMemoryIconInfo.imageOffsetInFile;
                        byte[] data = new byte[bytesToMove];
                        if (this.storageRandomAccessFileNextOffset != imageOffsetInFile) {
                            this.storageRandomAccessFile.seek(imageOffsetInFile);
                            this.storageRandomAccessFile.readFully(data);
                            this.storageRandomAccessFile.seek(this.storageRandomAccessFileNextOffset);
                            this.storageRandomAccessFile.write(data);
                        }
                        remappedOffset = this.storageRandomAccessFileNextOffset;
                        this.storageRandomAccessFileNextOffset += (long)bytesToMove;
                        Hash256 hash = (Hash256)this.hashByOffset.remove((Object)imageOffsetInFile);
                        Objects.requireNonNull(hash);
                        this.hashByOffset.put((Object)remappedOffset, (Object)hash);
                        offsetMap.put(imageOffsetInFile, remappedOffset);
                    }
                    catch (IOException e) {
                        throw new RuntimeException("IO exception while packing icons", e);
                    }
                }
                inMemoryIconInfo.imageOffsetInFile = remappedOffset;
            });
        }
    }

    synchronized void runHealthCheck() throws IOException {
        HashMap<Long, Hash256> actualHashByOffset = new HashMap<Long, Hash256>();
        HashMap<Hash256, Integer> actualHashUsesCount = new HashMap<Hash256, Integer>();
        ConcurrentMap cacheAsMap = this.memoryCache.asMap();
        HashSet<Long> cacheOffsetsChecked = new HashSet<Long>();
        for (InMemoryIconInfo inMemoryIconInfo : this.inMemoryIconInfos.values()) {
            BufferedImage bufferedImageFromDisk = this.loadFromDisk(inMemoryIconInfo);
            byte[] imageBytes = IconStorage.getImageBytes(bufferedImageFromDisk);
            Hash256 imageHash = IconStorage.getImageHash(imageBytes);
            Hash256 previousHashAtOffset = actualHashByOffset.put(inMemoryIconInfo.imageOffsetInFile, imageHash);
            if (previousHashAtOffset != null && !previousHashAtOffset.equals(imageHash)) {
                throw new RuntimeException("Different hashes at the same offset");
            }
            actualHashUsesCount.compute(imageHash, (k, count) -> count == null ? 1 : count + 1);
            BufferedImage imageFromCache = (BufferedImage)cacheAsMap.get(inMemoryIconInfo.imageOffsetInFile);
            if (imageFromCache == null) continue;
            byte[] imageFromCacheBytes = IconStorage.getImageBytes(imageFromCache);
            if (!Arrays.equals(imageFromCacheBytes, imageBytes)) {
                throw new RuntimeException("Cached image does not match the image on disk");
            }
            cacheOffsetsChecked.add(inMemoryIconInfo.imageOffsetInFile);
        }
        if (!cacheOffsetsChecked.equals(cacheAsMap.keySet())) {
            throw new RuntimeException("Checked offsets don't match cached offsets");
        }
        if (!this.hashByOffset.equals(actualHashByOffset)) {
            throw new RuntimeException("Hashes by offset don't match");
        }
        if (!this.hashUsesCount.equals(actualHashUsesCount)) {
            throw new RuntimeException("Hash uses don't match");
        }
    }

    synchronized int getUniqueImagesCount() {
        return this.hashUsesCount.size();
    }

    synchronized int getStoredImagesCount() {
        return this.inMemoryIconInfos.size();
    }

    synchronized long getUsedFileSize() {
        return this.storageRandomAccessFileNextOffset;
    }

    public static class IconInfo {
        int imageId;
        OnlineCalculatable.Marker marker;
    }

    private static class InMemoryIconInfo {
        double markerY;
        int iconOffsetX;
        int iconOffsetY;
        public long imageOffsetInFile;
        public int imageWidth;
        public int imageHeight;

        private InMemoryIconInfo() {
        }
    }

    private static class Hash256 {
        byte[] hash;

        public Hash256(byte[] data) {
            try {
                MessageDigest digest = MessageDigest.getInstance("SHA-512");
                this.hash = digest.digest(data);
            }
            catch (NoSuchAlgorithmException e) {
                throw new RuntimeException(e);
            }
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + Arrays.hashCode(this.hash);
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            Hash256 other = (Hash256)obj;
            return Arrays.equals(this.hash, other.hash);
        }
    }
}

